#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <GL/glui.h>
#include "parser.h"

float get_default_thickness(void);
float get_default_border(void);
float get_default_infinite(void);


/* ++++++++++++++++++++++++++ Test routines ++++++++++++++++++++++++++ */

void draw_neutron_track(int icomp, float x1, float y1, float z1,
					float x2, float y2, float z2)
{
  int i1,i2;
  float pos3[3], rot3x3_mx[9], rot4x4_mx[16];

/***** Save the matrix onto the stack *****/
  glPushMatrix();

/***** Try to get the named component *****/
  get_component(icomp);

/***** Get position and rotation of component *****/
  get_component_pos(pos3);
  get_component_rot(rot3x3_mx);

/***** Convert the rotation matrix from 3 x 3 to 4 x 4 *****/
  for(i1=0; i1<3; i1++) {
    for(i2=0; i2<3; i2++) {
      rot4x4_mx[4*i1+i2]=rot3x3_mx[3*i1+i2];
    }
    rot4x4_mx[4*i1+3]=0.0;
    rot4x4_mx[4*3+i1]=0.0;
  }
  rot4x4_mx[4*3+3]=1.0;

/***** Apply translation and rotation to matrix *****/
  glTranslatef(pos3[0],pos3[1],pos3[2]);
  glMultMatrixf(rot4x4_mx);

/***** Draw trace as a yellow line *****/
  glDisable( GL_LIGHTING );
  glBegin( GL_LINES );
   glColor3f( 1.0, 1.0, 0.0 );
  glVertex3f( x1, y1, z1 );
  glVertex3f( x2, y2, z2 );
  glEnd();
  glEnable( GL_LIGHTING );

/***** Restore the original matrix from the stack *****/
  glPopMatrix();
}

/* ####################################################################
   ##                                                                ##
   ##         Low level routines to draw simple shapes               ##
   ##                                                                ##
   #################################################################### */

/********************************************** draw_axes() **************/
void draw_axes(float xlen, float ylen, float zlen)
{
  glPushMatrix();

  glDisable( GL_LIGHTING );
  glBegin( GL_LINES );
 
/***** Red X axis *****/
  glColor3f( 1.0, 0.0, 0.0 );
  glVertex3f( 0.0, 0.0, 0.0 );
  glVertex3f( xlen, 0.0, 0.0 );

/***** Green Y axis *****/
  glColor3f( 0.0, 1.0, 0.0 );
  glVertex3f( 0.0, 0.0, 0.0 );
  glVertex3f( 0.0, ylen, 0.0 );

/***** Blue Z axis *****/
  glColor3f( 0.0, 0.0, 1.0 );
  glVertex3f( 0.0, 0.0, 0.0 );
  glVertex3f( 0.0, 0.0, zlen );

  glEnd();

  glEnable( GL_LIGHTING );
  glPopMatrix();
}


/********************************************** draw_bent_tube() **********/


void draw_bent_tube_segment(float x_depth, float y_width, float *angles, float rad_inner)
{
/***** Calculate X & Z coords for inner arc for both angles *****/
  float x_inner1=rad_inner*(1.0-cos(angles[0]))+x_depth/2.0;
  float z_inner1=rad_inner*sin(angles[0]);
  float x_inner2=rad_inner*(1.0-cos(angles[1]))+x_depth/2.0;
  float z_inner2=rad_inner*sin(angles[1]);

/***** Calculate X & Z coords for outer arc from the inner arc values *****/
  float x_outer1=x_inner1-x_depth*(1.0-x_inner1/rad_inner);
  float z_outer1=z_inner1*(1.0+x_depth/rad_inner);
  float x_outer2=x_inner2-x_depth*(1.0-x_inner2/rad_inner);
  float z_outer2=z_inner2*(1.0+x_depth/rad_inner);

/*****  Draw the QUADS to make one tube segment *****/
  glBegin( GL_QUAD_STRIP );

  glVertex3f( x_inner1, -y_width/2.0, z_inner1 );
  glVertex3f( x_inner2, -y_width/2.0, z_inner2 );

  glNormal3f( 0.0, -1.0, 0.0 );
  glVertex3f( x_outer1, -y_width/2.0, z_outer1 );
  glVertex3f( x_outer2, -y_width/2.0, z_outer2 );

  glNormal3f( -cos(angles[0]), 0.0, sin(angles[0]) );
  glVertex3f( x_outer1, +y_width/2.0, z_outer1 );
  glVertex3f( x_outer2, +y_width/2.0, z_outer2 );

  glNormal3f( 0.0,  1.0, 0.0 );
  glVertex3f( x_inner1, +y_width/2.0, z_inner1 );
  glVertex3f( x_inner2, +y_width/2.0, z_inner2 );

  glNormal3f( cos(angles[0]), 0.0, -sin(angles[0]) );
  glVertex3f( x_inner1, -y_width/2.0, z_inner1 );
  glVertex3f( x_inner2, -y_width/2.0, z_inner2 );

  glEnd();
}


void draw_bent_tube(float x_depth, float y_width, float angle, float radius)
{
  int i,nangles;
  float angles[2];

  nangles=angle/0.05;
  if(nangles < 4) nangles=4;

  for(i=1; i<nangles; i++) {
    angles[0]=angle*(i-1)/float(nangles-1);
    angles[1]=angle*i/float(nangles-1);
    draw_bent_tube_segment(x_depth, y_width, angles, radius-x_depth/2.0);
  }
}



/******************************************** draw_taper_tube() ********/

void draw_taper_tube(float x_depth1,float y_width1,
                     float x_depth2,float y_width2, float z_length)
{
  glBegin( GL_QUAD_STRIP );

  glVertex3f( x_depth1/2.0, -y_width1/2.0, 0.0 );
  glVertex3f( x_depth2/2.0, -y_width2/2.0, z_length );

  glNormal3f( 0.0,-1.0, 0.0 );
  glVertex3f(-x_depth1/2.0, -y_width1/2.0, 0.0 );
  glVertex3f(-x_depth2/2.0, -y_width2/2.0, z_length );

  glNormal3f(-1.0, 0.0, 0.0 );
  glVertex3f(-x_depth1/2.0, +y_width1/2.0, 0.0 );
  glVertex3f(-x_depth2/2.0, +y_width2/2.0, z_length );

  glNormal3f( 0.0,  1.0, 0.0 );
  glVertex3f( x_depth1/2.0, +y_width1/2.0, 0.0 );
  glVertex3f( x_depth2/2.0, +y_width2/2.0, z_length );

  glNormal3f( 1.0, 0.0, 0.0 );
  glVertex3f( x_depth1/2.0, -y_width1/2.0, 0.0 );
  glVertex3f( x_depth2/2.0, -y_width2/2.0, z_length );

  glEnd();
}



/******************************************** draw_straight_tube() ********/


void draw_straight_tube(float x_depth, float y_width, float z_length)
{
  glBegin( GL_QUAD_STRIP );

  glVertex3f( x_depth/2.0, -y_width/2.0, 0.0 );
  glVertex3f( x_depth/2.0, -y_width/2.0, z_length );

  glNormal3f( 0.0,-1.0, 0.0 );
  glVertex3f(-x_depth/2.0, -y_width/2.0, 0.0 );
  glVertex3f(-x_depth/2.0, -y_width/2.0, z_length );

  glNormal3f(-1.0, 0.0, 0.0 );
  glVertex3f(-x_depth/2.0, +y_width/2.0, 0.0 );
  glVertex3f(-x_depth/2.0, +y_width/2.0, z_length );

  glNormal3f( 0.0,  1.0, 0.0 );
  glVertex3f( x_depth/2.0, +y_width/2.0, 0.0 );
  glVertex3f( x_depth/2.0, +y_width/2.0, z_length );

  glNormal3f( 1.0, 0.0, 0.0 );
  glVertex3f( x_depth/2.0, -y_width/2.0, 0.0 );
  glVertex3f( x_depth/2.0, -y_width/2.0, z_length );

  glEnd();
}



/******************************************** draw_cylinder_tube() ********/


void draw_cylinder_tube(float radius, float z_length)
{
  int iangle, nangles=20;
  float angle;

  glBegin( GL_QUAD_STRIP );

  for(iangle=0; iangle<nangles; iangle++) {
    angle=2.0*3.1415927*iangle/(nangles-1);
    glNormal3f( cos(angle), sin(angle), 0.0 );
    glVertex3f( radius*cos(angle), radius*sin(angle), 0.0      );
    glVertex3f( radius*cos(angle), radius*sin(angle), z_length );
  }

  glEnd();
}



/****************************************** draw_square_aperture() ********/


void draw_square_aperture(float x_height, float y_width, float x_border, float y_border,
									float z_thickness)
{
  int i;
  float z;

/***** Draw QUADS to make both faces of aperture *****/
  for(i=0; i<2; i++) {
    glBegin( GL_QUAD_STRIP );

/***** Make z=0 face on first pass, z>0 face on second pass *****/
    z=0.0;
    glNormal3f( 0.0, 0.0,-1.0 );
    if(i == 1) {
      z=z_thickness;
      glNormal3f( 0.0, 0.0, 1.0 );
    }

/***** -X quad *****/
    glVertex3f( -x_height/2.0,          -y_width/2.0,          z );
    glVertex3f( -x_height/2.0-x_border, -y_width/2.0-y_border, z );
    glVertex3f( -x_height/2.0,           y_width/2.0,          z );
    glVertex3f( -x_height/2.0-x_border,  y_width/2.0+y_border, z );

/***** +Y quad *****/
    glVertex3f(  x_height/2.0,           y_width/2.0,          z );
    glVertex3f(  x_height/2.0+x_border,  y_width/2.0+y_border, z );

/***** +X quad *****/
    glVertex3f(  x_height/2.0,          -y_width/2.0,          z );
    glVertex3f(  x_height/2.0+x_border, -y_width/2.0-y_border, z );

/***** -Y quad *****/
    glVertex3f( -x_height/2.0,          -y_width/2.0,          z );
    glVertex3f( -x_height/2.0-x_border, -y_width/2.0-y_border, z );

  glEnd();
  }

/***** Add tubes to cover inner and outer edges of aperture *****/
  if(x_height!=0.0 && y_width!=0.0)
    draw_straight_tube(-x_height, -y_width, z_thickness);
  draw_straight_tube(x_height+2.0*x_border, y_width+2.0*y_border, z_thickness);
}



/******************************************* draw_round_aperture() ********/

void draw_round_aperture(float radius, float x_border, float y_border, float z_thickness)
{
  int iside, iedge, iangle, nangles=10;
  float x_inner,y_inner, x_outer,y_outer, z, outer_distance;

/***** Draw QUADS to make both sides of aperture *****/
  for(iside=0; iside<2; iside++) {
    glBegin( GL_QUAD_STRIP );

/***** Make z=0 face on first pass, z>0 face on second pass *****/
    z=0.0;
    glNormal3f( 0.0, 0.0,-1.0 );
    if(iside == 1) {
      z=z_thickness;
      glNormal3f( 0.0, 0.0, 1.0 );
    }

/***** We start at the -ve X & Y corner of the square outer border,
   and move from edge to edge around the the border			*/
    x_outer=-radius-x_border;
    y_outer=-radius-y_border;
    for(iedge=0; iedge<4; iedge++) {

/***** Move along each edge in "nangles" steps *****/
      for(iangle=0; iangle<nangles-1; iangle++) {

/***** Calculate X & Y on circular aperature at same angle as X & Y on edge *****/
        outer_distance=sqrt( x_outer*x_outer + y_outer*y_outer );
        x_inner=x_outer*radius/outer_distance;
        y_inner=y_outer*radius/outer_distance;

/***** Make a pair of vertices for QUAD from the points on edge and the aperture *****/
        glVertex3f( x_outer, y_outer, z );
        glVertex3f( x_inner, y_inner, z );

/***** Move the outer X & Y along the edge *****/
        if(iedge == 0)
          y_outer=y_outer+2.0*(radius+y_border)/(nangles-1);
        else if(iedge == 1)
          x_outer=x_outer+2.0*(radius+x_border)/(nangles-1);
        else if(iedge == 2)
          y_outer=y_outer-2.0*(radius+y_border)/(nangles-1);
        else
          x_outer=x_outer-2.0*(radius+x_border)/(nangles-1);

      }   /****** back for next step along edge *****/
    }     /****** back for next edge *****/

/***** Add the last QUAD pair to fill in the missing piece *****/
    outer_distance=sqrt( x_outer*x_outer + y_outer*y_outer );
    x_inner=x_outer*radius/outer_distance;
    y_inner=y_outer*radius/outer_distance;
    glVertex3f( x_outer, y_outer, z );
    glVertex3f( x_inner, y_inner, z );

/***** Finished with one side, loop back for the other *****/
    glEnd();
  }

/***** Add tubes to cover inner and outer edges of aperture *****/
  draw_straight_tube(2.0*(radius+x_border), 2.0*(radius+y_border), z_thickness);
  draw_cylinder_tube(-radius, z_thickness);
}


/**************************************** draw_hollow_cylinder() **********/

void draw_hollow_cylinder(float radius, float border, float z_length)
{
  int iside, iangle, nangles=40;
  float angle, z;

/***** Draw QUADS *****/
  for(iside=0; iside<2; iside++) {
    glBegin( GL_QUAD_STRIP );

/***** Make z=0 face on first pass, z>0 face on second pass *****/
    z=0.0;
    glNormal3f( 0.0, 0.0,-1.0 );
    if(iside == 1) {
      z=z_length;
      glNormal3f( 0.0, 0.0, 1.0 );
    }

    for(iangle=0; iangle<nangles; iangle++) {
      angle=2.0*3.1415926*iangle/(nangles-1);
      glVertex3f( cos(angle)*radius,          sin(angle)*radius,          z );
      glVertex3f( cos(angle)*(radius+border), sin(angle)*(radius+border), z );
    }

/***** Finished with one end, loop back for the other *****/
    glEnd();
  }

/***** Add tubes to join two end disks *****/
  draw_cylinder_tube(border+radius, z_length);
  if(radius != 0.0)
     draw_cylinder_tube(-radius, z_length);
}



/********************************************** draw_sphere() **********/

void draw_sphere(float radius)
{
  float theta_lo=0.0;
  float theta_hi=3.1415926;
  int iphi, itheta, nphi=15, ntheta=15;
  float theta1, theta2, phi, x1,y1,z1, x2,y2,z2;

/***** Draw QUADS *****/
  for(itheta=0; itheta<ntheta; itheta++) {
    theta1=theta_lo+(theta_hi-theta_lo)*itheta/ntheta;
    theta2=theta_lo+(theta_hi-theta_lo)*(itheta+1)/ntheta;

    glBegin( GL_QUAD_STRIP );

    for(iphi=0; iphi<nphi; iphi++) {
      phi=2.0*3.1415926*iphi/(nphi-1);
      z1=radius*cos(theta1);
      x1=radius*sin(theta1)*sin(phi);
      y1=radius*sin(theta1)*cos(phi);
      z2=radius*cos(theta2);
      x2=radius*sin(theta2)*sin(phi);
      y2=radius*sin(theta2)*cos(phi);
      glNormal3f( x1+x2, y1+y2, z1+z2 );
      glVertex3f( x1, y1, z1 );
      glVertex3f( x2, y2, z2 );
    }

    glEnd();
  }
}


/**************************************** draw_surface_circle() **********/

void draw_surface_circle(float radius)
{
  int iangle, nangles=40;
  float angle;

  glBegin( GL_QUAD_STRIP );
  glNormal3f( 0.0, 0.0, 1.0 );

  for(iangle=0; iangle<nangles; iangle++) {
    angle=2.0*3.1415926*iangle/(nangles-1);
    glVertex3f( 0.0, 0.0, 0.0 );
    glVertex3f( cos(angle)*radius, sin(angle)*radius, 0.0 );
  }

  glEnd();
}



/************************************* draw_surface_rectangle() *********/

void draw_surface_rectangle(float xwidth, float yheight)
{
  int iangle, nangles=40;
  float angle;

  glBegin( GL_QUAD_STRIP );
  glNormal3f( 0.0, 0.0, 1.0 );

  glVertex3f( -xwidth/2.0, -yheight/2.0, 0.0 );
  glVertex3f( -xwidth/2.0, +yheight/2.0, 0.0 );
  glVertex3f( +xwidth/2.0, -yheight/2.0, 0.0 );
  glVertex3f( +xwidth/2.0, +yheight/2.0, 0.0 );

  glEnd();
}






/* ####################################################################
   ##                                                                ##
   ##     Mid level routine to draw objects in terms of shapes       ##
   ##                                                                ##
   #################################################################### */


/*************************************** draw_components() ****************/

/***** Draw an object of type "token" with arguments "args[nargs]" *****/
void draw_token_object(int token, float *args, int nargs)
{
  float thickness=get_default_thickness();
  float border=get_default_border();
  float infinite=get_default_infinite();

  if(token ==TOKEN_surface_infinite) {
    draw_surface_rectangle( infinite, infinite );
  }
  else if(token ==TOKEN_axes) {
    draw_axes( args[0], args[1], args[2] );
  }
  else if(token ==TOKEN_box) {
    if(args[0] == 0.0)
      draw_square_aperture( 0.0,0.0, thickness/2.0, args[1]/2.0, args[2] );
    else if(args[1] == 0.0)
      draw_square_aperture( 0.0,0.0, args[0]/2.0, thickness/2.0, args[2] );
    else if(args[2] == 0.0)
      draw_square_aperture( 0.0,0.0, args[0]/2.0, args[1]/2.0, thickness );
    else
      draw_square_aperture( 0.0,0.0, args[0]/2.0, args[1]/2.0, args[2] );
  }
  else if(token ==TOKEN_hollow_cylinder) {
    if(args[2] != 0.0)
       thickness=args[2];
    draw_hollow_cylinder( args[0], args[1], thickness );
  }
  else if(token ==TOKEN_hollow_cylinder_y) {
    if(args[2] != 0.0)
       thickness=args[2];
    glRotatef(180.0, 0.0,0.707,0.707);
    draw_hollow_cylinder( args[0], args[1], thickness );
  }
  else if(token ==TOKEN_slit_circle) {
    draw_round_aperture( args[0], border,border,thickness );
  }
  else if(token ==TOKEN_slit_rectangle) {
    draw_square_aperture( args[0], args[1], border,border,thickness );
  }
  else if(token ==TOKEN_surface_circle) {
    draw_surface_circle( args[0] );
  }
  else if(token ==TOKEN_surface_cylinder_y) {
    glRotatef(180.0, 0.0,0.707,0.707);
    draw_cylinder_tube( args[0], args[1] );
  }
  else if(token ==TOKEN_surface_rectangle) {
    draw_surface_rectangle( args[0], args[1] );
  }
  else if(token ==TOKEN_surface_sphere) {
    draw_sphere( args[0] );
  }
  else if(token ==TOKEN_tube) {
    draw_straight_tube( args[0], args[1], args[2] );
  }
  else if(token ==TOKEN_tube_taper) {
    draw_taper_tube( args[0], args[1], args[2], args[3], args[4] );
  }
}



void set_render_values(float *args)
{
  int i;
  float ambient[4], diffuse[4], specular[4];

  for(i=0; i<3; i++) {
    ambient[i] =args[i]*(       args[3] )*2.0;
    diffuse[i] =args[i]*( 1.0 - args[3] )*2.0;
    specular[i]=args[4];
  }
  ambient[3]=1.0;
  diffuse[3]=1.0;
  specular[3]=1.0;

  glMaterialfv(GL_FRONT, GL_AMBIENT,   ambient);
  glMaterialfv(GL_FRONT, GL_DIFFUSE,   diffuse);
  glMaterialfv(GL_FRONT, GL_SPECULAR,  specular);
  glMaterialf (GL_FRONT, GL_SHININESS, 10.0);

/***** If opacity (args[5]) > 0.5 turn off stipple, else turn on *****/
  if(args[5] > 0.5)
    glDisable(GL_POLYGON_STIPPLE);
  else
    glEnable(GL_POLYGON_STIPPLE);
}



/* ####################################################################
   ##                                                                ##
   ## High level routine to draw component as a list of draw objects ##
   ##                                                                ##
   #################################################################### */

/***** Draw the current component in the component list. *****/
void draw_current_component(void)
{
  int i1,i2, token, nargs, iobject,nobjects;
  float args[7], pos3[3], rot3x3_mx[9], rot4x4_mx[16];

/***** Save the matrix onto the stack *****/
  glPushMatrix();

/***** Get position and rotation of component *****/
  get_component_pos(pos3);
  get_component_rot(rot3x3_mx);

/***** Convert the rotation matrix from 3 x 3 to 4 x 4 *****/
  for(i1=0; i1<3; i1++) {
    for(i2=0; i2<3; i2++) {
      rot4x4_mx[4*i1+i2]=rot3x3_mx[3*i1+i2];
    }
    rot4x4_mx[4*i1+3]=0.0;
    rot4x4_mx[4*3+i1]=0.0;
  }
  rot4x4_mx[4*3+3]=1.0;

/***** Apply translation and rotation to matrix *****/
  glTranslatef(pos3[0],pos3[1],pos3[2]);
  glMultMatrixf(rot4x4_mx);

/***** Save matrix for use by all draw objects in component *****/
  glPushMatrix();

/***** Loop through all draw objects in the draw list *****/
  nobjects=get_num_draw_objects();
  for(iobject=0; iobject<nobjects; iobject++) {
    get_draw_object(iobject+1);

/***** Load token and arguments for draw object *****/
    token=get_draw_object_token();
    nargs=get_draw_object_args(args);

/***** If a "move", then reload copied matrix and add translation *****/
    if(token == TOKEN_move) {
      glPopMatrix();
      glPushMatrix();
      glTranslatef( args[0], args[1], args[2] );
    }

/***** Add rendering information, i.e. colour and lighting *****/
    else if(token == TOKEN_render) {
      set_render_values(args);
    }

/***** Must be a real object, so draw it *****/
    else
      draw_token_object(token,args,nargs);

  } /****** loop back for next draw object *****/

/***** Finished with component so pop copied matrix off stack *****/
  glPopMatrix();

/***** Restore the original matrix from the stack *****/
  glPopMatrix();
}



/* ####################################################################
   ##                                                                ##
   ##            High level routine to draw all components           ##
   ##                                                                ##
   #################################################################### */

/***** Draw all components in component list that have visibility on *****/
void draw_all_components(int *visibility)
{
  int i, n;

/***** Always draw axes at the origin *****/
  draw_axes( 0.5, 0.5, 1.0 );

/***** Draw all components in list *****/
  n=get_num_components();
  for(i=0; i<n; i++) {
    if( visibility[i] ) {
      get_component(i+1);
      draw_current_component();
    }
  }

}



struct mcstas_neutron_track {
    int component;
    float x1,y1,z1, x2,y2,z2;
};


void draw_neutron_tracks(struct mcstas_neutron_track *tracks, int ntracks)
{
  int i;

  for(i=0; i<ntracks; i++) {
    draw_neutron_track( tracks[i].component,
                        tracks[i].x1,tracks[i].y1,tracks[i].z1,
                        tracks[i].x2,tracks[i].y2,tracks[i].z2  );
  }
}


