#include <string.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <GL/glui.h>
#include "mc2d.h"


/***** Structures needed for storing monitor data *****/

/* A list of up to 1000 2D monitor data sets, the same
   hardwired limit of 1000 is also used in read_mon.cpp */
struct mcstas_mon_file *mon_2d_files_list[1000];
int num_mon_2d_files=0;

/* Structure to store 2D monitor data that is being displayed */
struct mc2d_image_2d image_2d[1];



/***** Various GL and GLUT related variables *****/

/* Identifier for the main window */
int   main_window;

/* Lists of GL commands for drawing various objects */
static GLint plate_list, contours_list, blocks_list;
static GLint axes_list,legend_list;



/***** Various GLUI related variables *****/

/* Pointers needed globally */
GLUI *glui;
GLUI_Spinner *plot_index_xlo_spinner;
GLUI_Spinner *plot_index_xhi_spinner;
GLUI_Spinner *plot_index_ylo_spinner;
GLUI_Spinner *plot_index_yhi_spinner;
GLUI_Spinner *plot_intensity_spinner;

/* Overall scale and rotation of objects */
float scale = 1.0;
float view_rotate[16] = { 1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1 };

/* Spinners values for displaying data */
int plot_index_xlo, plot_index_xhi;
int plot_index_ylo, plot_index_yhi;
float plot_intensity;

/* Selection values for list boxes */
int monitor_select=0;
#define COLOUR_MAP_BLUE_RED    0
#define COLOUR_MAP_GREY_1      1
#define COLOUR_MAP_GREY_2      2
int colour_map_select=COLOUR_MAP_BLUE_RED;
#define DISPLAY_MODE_HISTOGRAM     0
#define DISPLAY_MODE_SIMPLE        1
#define DISPLAY_MODE_CONTOURS_5    2
#define DISPLAY_MODE_CONTOURS_10   3
int display_mode_select=DISPLAY_MODE_HISTOGRAM;


/***********************************************************************
 *                                                                     *
 *         Routines to load, save and restore 2D monitor data          *
 *                                                                     *
 ***********************************************************************/


/* Load monitor data from files into mon_2d_files_list[] */
void load_mon_2d_files_list(void)
{
  int i, n0,n1,n2;

/***** Read in all monitor files in mcstas simulation file *****/
  if( !init_mon_files(&n0,&n1,&n2) ) {
    printf("ERROR: Couldn't parse simulation file\n");
    exit(1);
  }

/***** Print out number of 2D files found, and give up if zero *****/
  printf("Total of %d monitors in simulation, ",n0+n1+n2);
  printf("of which %d are 2D\n",n2);
  if(n2 == 0) {
    printf("\nWARNING: Program exits as nothing to display\n");
    exit(1);
  }

/***** Load all 2D monitor file data into mon_2d_files_list[] *****/
  for(i=0; i<n2; i++) {
    mon_2d_files_list[i]=get_2d_mon_file(i);
    if(mon_2d_files_list[i] == NULL) {
      printf("ERROR: Unable to load 2d monitor file #%d\n",i+1);
      exit(1);
    }
  }

/***** Store the number of 2D monitor files in the list *****/
  num_mon_2d_files=n2;
}



/* Load monitor data from mon_2d_files_list[] into *p,
   setting the X, Y & Intensity spinners to the data limits */
void load_image_2d_data(struct mc2d_image_2d *p, int imon)
{
  int ix,iy, n0,n1,n2, idx;
  struct mcstas_mon_file *mon=mon_2d_files_list[imon];

/***** Print out number of X & Y values *****/
  printf("nx & ny = %d %d\n", mon->nx,mon->ny);

/***** Copy needed values and pointers to *p structure *****/
  p->x_width=mon->nx;
  p->y_height=mon->ny;
  p->component=mon->component;
  p->x_label=mon->x_label;
  p->y_label=mon->y_label;
  p->xlo=mon->xlo;
  p->xhi=mon->xhi;
  p->ylo=mon->ylo;
  p->yhi=mon->yhi;

/***** Free up storage for p->data, then allocate new memory *****/
  free(p->data);
  p->data=(float *)malloc( (mon->nx) * (mon->ny) * sizeof(float) );

/***** Copy data, and calculate and print out maximum value *****/
  p->data_max=0.0;
  for(iy=0; iy<(p->y_height); iy++) {
    for(ix=0; ix<(p->x_width); ix++) {
      idx=iy*(p->x_width) + ix;
      p->data[idx]=mon->data[idx];
      p->data_max=MAX(p->data_max, p->data[idx] );
    }
  }
  printf("Maximum intensity value = %g\n",p->data_max);

/***** If data_max == 0 reset to 1 to prevent div-zero crash *****/
  if(p->data_max == 0)
    p->data_max=1.0;

/***** Zero the contour parameters *****/
  p->nlines=0;
  p->lines=NULL;

/***** Set slider values for X limits *****/
  plot_index_xlo_spinner->set_int_limits( 1, p->x_width-1 );
  plot_index_xlo=1;
  plot_index_xlo_spinner->set_int_val( 1 );
  plot_index_xhi_spinner->set_int_limits( 2, p->x_width );
  plot_index_xhi=p->x_width;
  plot_index_xhi_spinner->set_int_val( p->x_width );

/***** Set slider values for Y limits *****/
  plot_index_ylo_spinner->set_int_limits( 1, p->y_height-1 );
  plot_index_ylo=1;
  plot_index_ylo_spinner->set_int_val( 1 );
  plot_index_yhi_spinner->set_int_limits( 2, p->y_height );
  plot_index_yhi=p->y_height;
  plot_index_yhi_spinner->set_int_val( p->y_height );

/***** Set slider values for Intensities *****/
  plot_intensity_spinner->set_float_limits( 0.0, p->data_max);
  plot_intensity=p->data_max;
  plot_intensity_spinner->set_float_val( p->data_max);
}



/* Reload monitor data from mon_2d_files_list[] into *p,
   using the limits set by the X, Y & Intensity spinners */
void reload_image_2d_data(struct mc2d_image_2d *p, int imon)
{
  int ix,iy, n0,n1,n2, idx1, idx2;
  struct mcstas_mon_file *mon=mon_2d_files_list[imon];
  float dx,dy;

/***** Store number of points for X & Y in *p structure *****/
  p->x_width=plot_index_xhi-plot_index_xlo+1;
  p->y_height=plot_index_yhi-plot_index_ylo+1;

/***** Store upper & lower limits for X & Y in *p structure *****/
  dx=(mon->xhi-mon->xlo)/mon->nx;
  p->xlo=mon->xlo + dx*(plot_index_xlo-1);
  p->xhi=mon->xhi - dx*(mon->nx - plot_index_xhi);
  dy=(mon->yhi-mon->ylo)/mon->ny;
  p->ylo=mon->ylo + dy*(plot_index_ylo-1);
  p->yhi=mon->yhi - dy*(mon->ny - plot_index_yhi);

/***** Free up storage for p->data, then allocate new memory *****/
  free(p->data);
  p->data=(float *)malloc( (p->x_width) * (p->y_height) * sizeof(float) );

/***** Copy data into p->data and calculate maximum data value *****/
  idx1=0; 
  for(iy=plot_index_ylo-1; iy<plot_index_yhi; iy++) {
    for(ix=plot_index_xlo-1; ix<plot_index_xhi; ix++) {
       idx2=iy*(mon->nx) + ix;
       p->data[idx1++]=mon->data[idx2];
     }
   }

/***** Zero the contour parameters *****/
  p->nlines=0;
  p->lines=NULL;
}



/***********************************************************************
 *                                                                     *
 *      Routines that draw the data representation onto the plot       *
 *                                                                     *
 ***********************************************************************/

/* Calculate colours 'r', 'g' & 'b'  corresponding to 'value' */
void map_colour(float *r, float *g, float *b, float value)
{
/***** Clamp 'value' to between 0 & 1 *****/
  value=MAX(0.0,MIN(1.0, value ));

/***** Calculate r, g & b for the various colour maps *****/
  if(colour_map_select == COLOUR_MAP_GREY_1) {
    *r=value;
    *g=value;
    *b=value;
  }
  else if(colour_map_select == COLOUR_MAP_GREY_2) {
    *r=sqrt(value);
    *g=sqrt(value);
    *b=sqrt(value);
  }
  else {
    *r=value;
    *g=1.0-SQR(SQR(1.0-2.0*value));
    *b=1.0-value;
  }
}


void draw_contours(struct mc2d_image_2d *p_image)
{
  int i;
  float z;

  glBegin(GL_LINES);
  glColor3f(0.0,0.0,0.0);

  for(i=0; i<p_image->nlines; i++) {
    glVertex3f( (p_image->lines+i)->x1,
		(p_image->lines+i)->y1,  0.001);
    glVertex3f( (p_image->lines+i)->x2,
		(p_image->lines+i)->y2,  0.001);
  }

  glEnd();
}



void draw_plate(struct mc2d_image_2d *p)
{
  int ix,iy, idx;
  float z, r,g,b, dx,dy;

/* Calculate size of rectangles per point */
  dx=1.0/(float )(p->x_width);
  dy=1.0/(float )(p->y_height);

/* Start looping through lines of data */
  for(iy=0; iy<(p->y_height); iy++) {

/* Start stepping through x values in this line */
    for(ix=0; ix<(p->x_width); ix++) {

/* Set colour by the z value of the point */
      idx=iy*(p->x_width)+ix;
      if(p->data_max == 0.0)
        z=1.0;
      else
        z=MIN(1.0, *(p->data+idx) / p->data_max );

/* Give up on this point if z is too small */
      if(z < 0.001)
        continue;

/* Set colour for data point */
      map_colour(&r,&g,&b, z);
      glColor3f( r, g, b );

      glBegin(GL_QUAD_STRIP);
      glVertex3f( (ix  )*dx, (iy  )*dy, 0.0);
      glVertex3f( (ix  )*dx, (iy+1)*dy, 0.0);
      glVertex3f( (ix+1)*dx, (iy  )*dy, 0.0);
      glVertex3f( (ix+1)*dx, (iy+1)*dy, 0.0);
      glEnd();

    } /* end of loop for x steps in line */

  }
}



void draw_height_blocks(struct mc2d_image_2d *p)
{
  int ix,iy, idx;
  float z, r,g,b, dx,dy;

/* Calculate size of rectangles per point */
  dx=1.0/(float )(p->x_width);
  dy=1.0/(float )(p->y_height);

/* Loop through all point in data */
  for(iy=0; iy<(p->y_height); iy++) {
    for(ix=0; ix<(p->x_width); ix++) {

/* Calculate the z value normalized to data_max */
      idx=iy*(p->x_width)+ix;
      if(p->data_max == 0.0)
        z=1.0;
      else
        z=MIN(1.0, *(p->data+idx) / p->data_max );

/* Give up on this point if Z is too small */
      if(z < 0.001)
        continue;

/* Set colour by the z value of the point */
      map_colour(&r,&g,&b, z);
      glColor3f( r, g, b );

/* Make sides of block */
      glBegin(GL_QUAD_STRIP);
      glVertex3f( (ix  )*dx, (iy  )*dy, 0.0);
      glVertex3f( (ix  )*dx, (iy  )*dy, 0.25*z);
      glVertex3f( (ix  )*dx, (iy+1)*dy, 0.0);
      glVertex3f( (ix  )*dx, (iy+1)*dy, 0.25*z);
      glVertex3f( (ix+1)*dx, (iy+1)*dy, 0.0);
      glVertex3f( (ix+1)*dx, (iy+1)*dy, 0.25*z);
      glVertex3f( (ix+1)*dx, (iy  )*dy, 0.0);
      glVertex3f( (ix+1)*dx, (iy  )*dy, 0.25*z);
      glVertex3f( (ix  )*dx, (iy  )*dy, 0.0);
      glVertex3f( (ix  )*dx, (iy  )*dy, 0.25*z);
      glEnd();

/* Make top of block */
      glBegin(GL_QUADS);
      glVertex3f( (ix  )*dx, (iy  )*dy, 0.25*z);
      glVertex3f( (ix  )*dx, (iy+1)*dy, 0.25*z);
      glVertex3f( (ix+1)*dx, (iy+1)*dy, 0.25*z);
      glVertex3f( (ix+1)*dx, (iy  )*dy, 0.25*z);
      glEnd();

    } /* end of loop for ix */
  } /* end of loop for iy */
}




/***********************************************************************
 *                                                                     *
 *      Routines for annotating plot with axes, ticks and legend       *
 *                                                                     *
 ***********************************************************************/

/* General utility to write 'string' at position (x,y,0)
   and 'angle' to the horizontal, and at size 'scale'      */
void draw_text(float x, float y, float angle, float scale,
                                               char *string)
{
/***** Save orientation matrix *****/
  glPushMatrix();

/***** Perform translation, rotation and scale *****/
  glTranslatef(x,y,0.0);
  glRotatef(angle, 0.0,0.0,1.0);
  glScalef(scale*1e-3,scale*1e-3,scale*1e-3);

/***** Translate text to center it (May not be portable!) *****/
  glTranslatef(-37.0*strlen(string),-50.0*scale,0.0);

/***** Write out 'string' one character at a time *****/
  while(*string) {
    glutStrokeCharacter(GLUT_STROKE_ROMAN, (int )(*string));
    ++string;
  }

/***** Restore matrix *****/
  glPopMatrix();
}



/* For data limits of 'x1' & 'x2' load suitable tick points
   into 'ticks[]' and return the number of tick points used */
int calc_ticks(float *ticks, float x1, float x2)
{
  float dx, tick;
  int ntick, n1,n2, i;

/***** Calculate a decimal multiple for the plot 'tick' *****/
  dx=x2-x1;
  tick=pow( 10.0, floor(log10(dx)) );

/***** Divide by 2 or 5 to get about ~<10 ticks *****/
  ntick=dx/tick+1e-4;
  if(ntick == 1) {
    tick=tick*0.1;
    ntick=dx/tick+1e-4;
    if(ntick > 10)
      tick=tick*2.0;
  }
  else if(ntick == 2) {
    tick=tick*0.2;
    ntick=dx/tick+1e-4;
    if(ntick > 10)
      tick=tick*2.5;
  }
  else if(ntick <= 5) {
    tick=tick*0.5;
    ntick=dx/tick+1e-4;
    if(ntick > 10)
      tick=tick*2.0;
  }

/***** Fill ticks[] with multiples of 'tick' between x1 & x2 *****/
  n1=ceil(x1/tick-1e-4);
  n2=floor(x2/tick+1e-4);
  for(i=n1; i<=n2; i++)
    ticks[i-n1]=i*tick;

/***** Return the number of tick points loaded into ticks[] *****/
  return (n2-n1+1);
}


void draw_ticks(float xlo, float xhi, float ylo, float yhi)
{
  char string[32];
  int nxticks, nyticks, i;
  float xticks[21],yticks[21], x,y;

/***** Calculate the ticks points we want to use for X & Y *****/
  nxticks=calc_ticks(xticks, xlo, xhi);
  nyticks=calc_ticks(yticks, ylo, yhi);

/***** Set GL to draw lines *****/
  glBegin(GL_LINES);

/***** Draw ticks on x axis *****/
  for(i=0; i<nxticks; i++) {
    x=xticks[i];
    x=(x-xlo)/(xhi-xlo);
    glVertex3f( x,0.0, 0.0);
    glVertex3f( x,-.01, 0.0);
  }

/***** Draw ticks on y axis *****/
  for(i=0; i<nyticks; i++) {
    y=yticks[i];
    y=(y-ylo)/(yhi-ylo);
    glVertex3f( 0.0,y, 0.0);
    glVertex3f(-.01,y, 0.0);
  }

/***** Draw grid lines spaced in x direction *****/
  for(i=0; i<nxticks; i++) {
    x=xticks[i];
    x=(x-xlo)/(xhi-xlo);
    glVertex3f( x,0.0, 0.0);
    glVertex3f( x,1.0, 0.0);
  }

/***** Draw grid lines spaced in y direction *****/
  for(i=0; i<nyticks; i++) {
    y=yticks[i];
    y=(y-ylo)/(yhi-ylo);
    glVertex3f( 0.0,y, 0.0);
    glVertex3f( 1.0,y, 0.0);
  }

/***** Turn off lines drawing *****/
  glEnd();

/***** Write first and last tick value for x axis *****/
  sprintf(string,"%g",xticks[0]);
  x=(xticks[0]-xlo)/(xhi-xlo);
  draw_text(x,-.05,  0.0, 0.25, string);
  sprintf(string,"%g",xticks[nxticks-1]);
  x=(xticks[nxticks-1]-xlo)/(xhi-xlo);
  draw_text(x,-.05,  0.0, 0.25, string);

/***** Write first and last tick value for y axis *****/
  sprintf(string,"%g",yticks[0]);
  y=(yticks[0]-ylo)/(yhi-ylo);
  draw_text(-.05,y,  90.0, 0.25, string);
  sprintf(string,"%g",yticks[nyticks-1]);
  y=(yticks[nyticks-1]-ylo)/(yhi-ylo);
  draw_text(-.05,y,  90.0, 0.25, string);
}


void draw_axes(struct mc2d_image_2d *p)
{
/***** Set GL for drawing black lines and text *****/
  glColor3f(0.0,0.0,0.0);

/***** Draw bounding line for plot *****/
  glBegin(GL_LINE_LOOP);
  glVertex3f( 0.0,0.0, 0.0);
  glVertex3f( 0.0,1.0, 0.0);
  glVertex3f( 1.0,1.0, 0.0);
  glVertex3f( 1.0,0.0, 0.0);
  glEnd();

/***** Draw the grid lines and X & Y tick marks on plot *****/
  draw_ticks(p->xlo,p->xhi,p->ylo,p->yhi);

/***** Write text for X & Y on plot *****/
  draw_text(-.07,0.5, 90.0, 0.4, p->y_label);
  draw_text(0.5,-.07,  0.0, 0.4, p->x_label);
}



void draw_legend(void)
{
  char string[8];
  float r,g,b, z;

/***** Draw colour map as a bar of 100 steps *****/
  glBegin(GL_QUAD_STRIP);
  for(z=0.0; z<1.0001; z=z+0.01) {
    map_colour(&r,&g,&b, z);
    glColor3f( r, g, b );
    glVertex3f( 0.70, -.60+0.50*z, 0.0);
    glVertex3f( 0.75, -.60+0.50*z, 0.0);
  }
  glEnd();

/***** Draw value in black from 0 to 1 in 0.1 steps *****/
  glColor3f( 0.0, 0.0, 0.0 );
  for(z=0.0; z<1.0001; z=z+0.1) {
    sprintf(string,"%.1f",z);
    draw_text(0.63,-.60+0.50*z, 0.0, 0.20, string);
  }
}



/***********************************************************************
 *                                                                     *
 *                       Callback routines                             *
 *                                                                     *
 ***********************************************************************/

void myGlutIdle( void )
{
/* For conformance to GLUT specification
  if(glutGetWindow() != main_window )
    glutSetWindow(main_window);  
  GLUI_Master.sync_live_all();               */
}


static void myGlutReshape(int w, int h)
{
  GLfloat aspect;
  int tx, ty, tw, th;

/***** Work out aspect ratio of display area *****/
  GLUI_Master.get_viewport_area( &tx, &ty, &tw, &th );
  glViewport( tx, ty, tw, th );
  aspect = (float)tw / (float)th;

/***** Set GL to correct aspect ratio *****/
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(-aspect, aspect, -1.0, 1.0, 4.0, 300.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glutPostRedisplay();
}



/***************************************** myGlutDisplay() **************/

void myGlutDisplay(void)
{
/***** Do the usual GL stuff *****/
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  glTranslatef( 0.0, 0.0, -10.0 );
  glScalef(4.0, 4.0, 4.0);
  glTranslatef( -0.05, 0.05, 0.0 );

/***** Scale all objects by 'scale' *****/
  glScalef( scale, scale, scale );

/***** Run the GL lists for objects that don't rotate *****/
  glCallList(legend_list);

/***** Rotate plot around center of data (0.5,0.5) *****/
  glMultMatrixf( view_rotate );
  glTranslatef( -.5, -.5, 0.0 );

/***** Run the GL lists for objects that rotate with the plot *****/
  glCallList(plate_list);
  glCallList(contours_list);
  glCallList(blocks_list);
  glCallList(axes_list);

/***** Recover saved matrix and swap the display buffers *****/
  glPopMatrix();
  glutSwapBuffers();
}



/*********************************** control panel callback *********/

void control_panel_cb( int iload )
{
/***** Load (or reload) image_2d from mon_2d_files_list[] *****/
  if(iload)
    load_image_2d_data(image_2d, monitor_select);
  else
    reload_image_2d_data(image_2d, monitor_select);

/***** Set the maximum intensity to plot *****/
  image_2d->data_max=plot_intensity;

/***** Calculate contour lines if required *****/
  if(display_mode_select == DISPLAY_MODE_CONTOURS_5)
    calc_contour_lines(image_2d, 5);
  else if(display_mode_select == DISPLAY_MODE_CONTOURS_10)
    calc_contour_lines(image_2d, 10);

/***** Make the GL list for the flat plate representation *****/
  glNewList(plate_list,GL_COMPILE);
  draw_plate(image_2d);
  glEndList();

/***** Make the GL list to draw the contour lines *****/
  glNewList(contours_list,GL_COMPILE);
  if(display_mode_select == DISPLAY_MODE_CONTOURS_5)
    draw_contours(image_2d);
  else if(display_mode_select == DISPLAY_MODE_CONTOURS_10)
    draw_contours(image_2d);
  glEndList();

/***** Make GL list to draw the histogram *****/
  glNewList(blocks_list,GL_COMPILE);
  if(display_mode_select == DISPLAY_MODE_HISTOGRAM)
    draw_height_blocks(image_2d);
  glEndList();


/***** Make GL list to draw the axes, ticks and grid lines *****/
  glNewList(axes_list,GL_COMPILE);
  draw_axes(image_2d);
  glEndList();


/***** Make GL list to legend for colourmap *****/
  glNewList(legend_list,GL_COMPILE);
  draw_legend();
  glEndList();


/***** Reset spinners to ensure you have at least 2 points to plot *****/
  plot_index_xlo_spinner->set_float_limits( 1, plot_index_xhi-1 );
  plot_index_xhi_spinner->set_float_limits( plot_index_xlo+1,
                                    mon_2d_files_list[monitor_select]->nx );
  plot_index_ylo_spinner->set_float_limits( 1, plot_index_yhi-1 );
  plot_index_yhi_spinner->set_float_limits( plot_index_ylo+1,

                                    mon_2d_files_list[monitor_select]->ny );

/***** Make sure display is updated *****/
  glutPostRedisplay();
}




/***********************************************************************
 *                                                                     *
 *    main() and the routine for creating the GLUI control panel       *
 *                                                                     *
 ***********************************************************************/

void create_control_panel(void)
{
  int i, n;

/**** Create subwindow on right of main window *****/
  glui=GLUI_Master.create_glui_subwindow(main_window,GLUI_SUBWINDOW_RIGHT);
  glui->set_main_gfx_window(main_window);

/**** Listbox for changing plot to different monitor components ****/
  monitor_select=0;
  GLUI_Listbox *monitor_listbox =
    glui->add_listbox( "Monitor:", &monitor_select, true, control_panel_cb);
  for(i=0; i<num_mon_2d_files; i++)
    monitor_listbox->add_item( i,  mon_2d_files_list[i]->component);

/***** Spinner for scaling objects *****/
  GLUI_Spinner *scale_spinner =
    glui->add_spinner("Scale:",
				GLUI_SPINNER_FLOAT, &scale);
  scale_spinner->set_float_limits( 0.01, 10.0 );

/***** "X data" panel with spinners for X start & X end *****/
  GLUI_Panel *x_data_panel = glui->add_panel( "X data" );
  plot_index_xlo_spinner = 
    glui->add_spinner_to_panel(x_data_panel,"Start",GLUI_SPINNER_INT,
                                &plot_index_xlo, false, control_panel_cb);
  plot_index_xlo_spinner->set_speed( 3.0 );
  plot_index_xhi_spinner = 
    glui->add_spinner_to_panel(x_data_panel,"End",GLUI_SPINNER_INT,
                                &plot_index_xhi, false, control_panel_cb);
  plot_index_xhi_spinner->set_speed( 3.0 );

/*****  "Y data" panel with spinners for Y start & Y end *****/
  GLUI_Panel *y_data_panel = glui->add_panel( "Y data" );
  plot_index_ylo_spinner = 
    glui->add_spinner_to_panel(y_data_panel,"Y start",GLUI_SPINNER_INT,
                                &plot_index_ylo, false, control_panel_cb);
  plot_index_ylo_spinner->set_speed( 3.0 );
  plot_index_yhi_spinner = 
    glui->add_spinner_to_panel(y_data_panel,"Y end",GLUI_SPINNER_INT,
                                &plot_index_yhi, false, control_panel_cb);
  plot_index_yhi_spinner->set_speed( 3.0 );

/***** "Intensity" panel with several controls *****/
  GLUI_Panel *intensity_panel = glui->add_panel( "Intensity" );
/* Spinner for intensity scale */
  plot_intensity_spinner = 
    glui->add_spinner_to_panel(intensity_panel,"Scale",GLUI_SPINNER_FLOAT,
                                &plot_intensity, false, control_panel_cb);
  plot_intensity_spinner->set_speed( 3.0 );
/* Listbox for different colour maps */
  GLUI_Listbox *colour_map_listbox =
    glui->add_listbox_to_panel(intensity_panel, "Colours",
                                &colour_map_select, false, control_panel_cb);
  colour_map_listbox->add_item(COLOUR_MAP_BLUE_RED,"Blue/Red");
  colour_map_listbox->add_item(COLOUR_MAP_GREY_1,"Grey 1");
  colour_map_listbox->add_item(COLOUR_MAP_GREY_2,"Grey 2");
/* Listbox for different display modes */
  GLUI_Listbox *display_mode_listbox =
    glui->add_listbox_to_panel(intensity_panel, "Display",
                                &display_mode_select, false, control_panel_cb);
  display_mode_listbox->add_item(DISPLAY_MODE_HISTOGRAM,"Histogram");
  display_mode_listbox->add_item(DISPLAY_MODE_SIMPLE,"Simple");
  display_mode_listbox->add_item(DISPLAY_MODE_CONTOURS_5,"Contours(5)");
  display_mode_listbox->add_item(DISPLAY_MODE_CONTOURS_10,"Contours(10)");

/***** Control for rotating view *****/
  GLUI_Rotation *view_rot =
    glui->add_rotation("Rotation", view_rotate );

/***** A 'quit' button *****/
  glui->add_button( "Quit", 0,(GLUI_Update_CB)exit );
}



/********************************************* main() *******/

int main(int argc, char *argv[])
{
/***** Turn off floating-point exceptions for BORLAND compiler *****/
#ifdef _BORLAND
  borland_GL_setup();
#endif

/***** Load pointer with NULL so we can safely use free() on it *****/
  image_2d->data=NULL;

/***** Load the 2D monitor data into mon_2d_files_list[] *****/
  load_mon_2d_files_list();

/***** Initialize GLUT *****/
  glutInitWindowPosition( 0, 0 );
  glutInitWindowSize(700, 450);
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
  main_window = glutCreateWindow("MC2D");
  
/***** Register GLUT & GLUI callbacks *****/
  glutDisplayFunc( myGlutDisplay );
  GLUI_Master.set_glutReshapeFunc( myGlutReshape );  

/***** More GL stuff *****/
  glDisable(GL_LIGHTING);
  glClearColor(0.3, 0.3, 0.6, 1.0);
  glEnable(GL_NORMALIZE);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LESS);

/***** Create GL lists for drawing objects *****/
  blocks_list=glGenLists(1);
  plate_list=glGenLists(1);
  contours_list=glGenLists(1);
  axes_list=glGenLists(1);
  legend_list=glGenLists(1);

/***** Create GLUI control panel *****/
  create_control_panel();

/***** Make sure displayed plot agrees with GLUI live values *****/
  control_panel_cb(true);

/***** Register the idle callback with GLUI, *not* with GLUT *****/
  GLUI_Master.set_glutIdleFunc( myGlutIdle );

/***** Let GLUT take control *****/
  glutMainLoop();
  return 0;         /* Unreachable code required by ANSI C */
}
