struct mcstas_neutron_track {
    int component;
    float x1,y1,z1, x2,y2,z2;
};


/*************************************************************************
 Contains high and low level routines for managing the lists of
 components and their draw object lists.

 The following are high level routines declared in "parser.h" and
 designed to be called by external routines:

 void component_parser(char *file_name)       input components into lists

 int get_num_components(void)                 # of components in list
 void get_component(int num)                  set current to 'num'
 int get_named_component(char *name)          true if "name" found
 char* get_component_name(void)               return pointer to string
 void get_component_pos(float *pos)           copies positions to pos[]
 void get_component_rot(float *rot)           copies 3x3 rot_mx to rot[]

 int get_num_draw_objects(void)               # of objects in draw list
 void get_draw_object(int num)                set draw object to 'num'
 int get_draw_object_token(void)              value defined in parser.h
 int get_draw_object_args(float *args)        returns nargs
*************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "parser.h"



/* ####################################################################
   ##                                                                ##
   ##      Low level routine to read a character line from a file    ##
   ##                                                                ##
   #################################################################### */

/* Read a line from 'file' into 'line_buffer[buffer_size]', ignoring
   any non-printable characters. Returns false if EOF is encountered. */
int read_line(char *line_buffer, int buffer_size, FILE *file)
{
  char c;
  int ic, nread=0;

/* Read characters to (almost) fill line_buffer */
  while(nread < buffer_size-1) {
/* If EOF or new-line, then we are finished reading */
    ic=getc(file);
    if(ic == EOF || ic == '\n')
      break;
/* Load any printable characters into line_buffer */
    c=(char )ic;
    if(c >= ' ')
      line_buffer[nread++]=c;
  }

/* Append zero termination to line_buffer */
  line_buffer[nread]='\0';

/* Return false if a bare EOF encountered */
  if(ic == EOF && nread == 0)
    return false;

/* All's right in the world, so return true */
  return true;
}



/* ####################################################################
   ##                                                                ##
   ##       Low level routines to manage component & draw lists      ##
   ##                                                                ##
   #################################################################### */

struct mc3d_component_list *component_list_head=NULL;
struct mc3d_component_list *component_list_current=NULL;


struct mc3d_component_list* add_component_to_list(char *name)
{
  struct mc3d_component_list *p_old=component_list_head, *p_new;

  if(p_old == NULL) {
    p_new=(struct mc3d_component_list *)
                       malloc(sizeof(struct mc3d_component_list));
    component_list_head=p_new;
  }
  else {
    while(p_old->next != NULL) {
      p_old=p_old->next;
    }
    p_new=(struct mc3d_component_list *)
                       malloc(sizeof(struct mc3d_component_list));
    p_old->next=p_new;
  }

  p_new->name=(char *)malloc(strlen(name));
  strcpy(p_new->name,name);
  p_new->draw_list=NULL;
  p_new->next=NULL;
  return p_new;
}


struct mc3d_component_list* search_component_list(char *search_name)
{
  struct mc3d_component_list *p=component_list_head;

  while(p != NULL) {
    if( 0 == strcmp(p->name, search_name) )
      break;
    p=p->next;
  }

  return p;
}


void switch_to_component(char *name)
{
  component_list_current=search_component_list(name);
  if(NULL == component_list_current)
    component_list_current=add_component_to_list(name);
}


void add_component_pos(float *pos)
{
  int i;

  if(component_list_current == NULL) {
     printf("ERROR: Tried to add POS without a current component\n");
     return;
  }

  for(i=0; i<3; i++)
    component_list_current->pos[i]=pos[i];
}


void add_component_rot(float *rot)
{
  int i;

  if(component_list_current == NULL) {
     printf("ERROR: Tried to add ROT without a current component\n");
     return;
  }

  for(i=0; i<9; i++)
    component_list_current->rot[i]=rot[i];
}


void add_component_draw_list(int token, float *f, int nargs)
{
  struct mc3d_draw_list *p_new, *p_old;

  if(component_list_current == NULL) {
     printf("ERROR: Tried to add DrawList without a current component\n");
     return;
  }

  p_old=component_list_current->draw_list;

  if(p_old == NULL) {
    p_new=(struct mc3d_draw_list *)malloc(sizeof(struct mc3d_draw_list));
    component_list_current->draw_list=p_new;
  }
  else {
    while(p_old->next != NULL) {
      p_old=p_old->next;
    }
    p_new=(struct mc3d_draw_list *)malloc(sizeof(struct mc3d_draw_list));
    p_old->next=p_new;
  }

  p_new->token=token;
  p_new->nargs=nargs;
  p_new->args[0]=f[0];
  p_new->args[1]=f[1];
  p_new->args[2]=f[2];
  p_new->args[3]=f[3];
  p_new->args[4]=f[4];
  p_new->args[5]=f[5];
  p_new->args[6]=f[6];
  p_new->next=NULL;
}



/* ####################################################################
   ##                                                                ##
   ##     Low level routines to read file and parse lines of file    ##
   ##                                                                ##
   #################################################################### */


/* Parses "line_buffer" for relevant MCDISPLAY lines. */
void parse_mcdisplay_line(char *line_buffer)
{
  char comp_string[80];
  int len;
  float pos[3], rot[9];

  if(1 == sscanf(line_buffer,"COMPONENT: \"%s",comp_string) ) {
    len=strlen(comp_string);
    if(comp_string[len-1] == '\"') {
      comp_string[len-1]='\0';
      switch_to_component(comp_string);
    }
  }
  else if(12 == sscanf( line_buffer,
                        "POS: %f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f",
                        &pos[0],&pos[1],&pos[2],
                        &rot[0],&rot[1],&rot[2],
                        &rot[3],&rot[4],&rot[5],
                        &rot[6],&rot[7],&rot[8]                     ) ) {
    add_component_pos(pos);
    add_component_rot(rot);
  }
  else if(1 == sscanf(line_buffer,"MCDISPLAY: component %s",comp_string) )
    switch_to_component(comp_string);
}



/* Parses "line_buffer" for valid MC3D lines. */
void parse_mc3d_line(char *line_buffer)
{
  int i, itoken, nargs;
  float *p, f[7];

  if( 0 == strcmp(line_buffer,"MC3D:surface_infinite()") ) {
                                        nargs=0;
                                        itoken=TOKEN_surface_infinite;
  }
  else if( 3 == (nargs=sscanf(line_buffer,"MC3D:axes(%f,%f,%f)",
                                  &f[0],&f[1],&f[2])) )
                                        itoken=TOKEN_axes;
  else if( 3 == (nargs=sscanf(line_buffer,"MC3D:box(%f,%f,%f)",
                                  &f[0],&f[1],&f[2])) )
                                        itoken=TOKEN_box;
  else if( 3 == (nargs=sscanf(line_buffer,"MC3D:hollow_cylinder(%f,%f,%f)",
                                  &f[0],&f[1],&f[2])) )
                                        itoken=TOKEN_hollow_cylinder;
  else if( 3 == (nargs=sscanf(line_buffer,"MC3D:hollow_cylinder_Y(%f,%f,%f)",
                                  &f[0],&f[1],&f[2])) )
                                        itoken=TOKEN_hollow_cylinder_y;
  else if( 3 == (nargs=sscanf(line_buffer,"MC3D:move(%f,%f,%f)",
                                  &f[0],&f[1],&f[2])) )
                                        itoken=TOKEN_move;
  else if( 6 == (nargs=sscanf(line_buffer,"MC3D:render(%f,%f,%f,%f,%f,%f)",
                                  &f[0],&f[1],&f[2],&f[3],&f[4],&f[5])) )
                                        itoken=TOKEN_render;
  else if( 1 == (nargs=sscanf(line_buffer,"MC3D:slit_circle(%f)",
                                  &f[0])) )
                                        itoken=TOKEN_slit_circle;
  else if( 2 == (nargs=sscanf(line_buffer,"MC3D:slit_rectangle(%f,%f)",
                                  &f[0],&f[1])) )
                                        itoken=TOKEN_slit_rectangle;
  else if( 1 == (nargs=sscanf(line_buffer,"MC3D:surface_circle(%f)",
                                  &f[0])) )
                                        itoken=TOKEN_surface_circle;
  else if( 2 == (nargs=sscanf(line_buffer,"MC3D:surface_cylinder_Y(%f,%f)",
                                  &f[0],&f[1])) )
                                        itoken=TOKEN_surface_cylinder_y;
  else if( 2 == (nargs=sscanf(line_buffer,"MC3D:surface_rectangle(%f,%f)",
                                  &f[0],&f[1])) )
                                        itoken=TOKEN_surface_rectangle;
  else if( 1 == (nargs=sscanf(line_buffer,"MC3D:surface_sphere(%f)",
                                  &f[0])) )
                                        itoken=TOKEN_surface_sphere;
  else if( 3 == (nargs=sscanf(line_buffer,"MC3D:tube(%f,%f,%f)",
                                  &f[0],&f[1],&f[2])) )
                                        itoken=TOKEN_tube;
  else if( 5 == (nargs=sscanf(line_buffer,"MC3D:tube_taper(%f,%f,%f,%f,%f)",
                                  &f[0],&f[1],&f[2],&f[3],&f[4])) )
                                        itoken=TOKEN_tube_taper;
  else {
    itoken=0;
    if( 0 == strncmp(line_buffer,"MC3D",4) )
      printf("Invalid MC3D line = %s\n",line_buffer);
  }

  if(itoken > 0)
    add_component_draw_list(itoken,f,nargs);
}



/* ####################################################################
   ##                                                                ##
   ##   High level set of routines to manage component & draw lists  ##
   ##                                                                ##
   #################################################################### */


/************************** Component List Routines *********************/

 int get_num_components(void)
{
  int i=0;
  struct mc3d_component_list *p;

  p=component_list_head;
  while(p != NULL) {
    i++;
    p=p->next;
  }

  return i;
}


 void get_component(int num)
{
  int i;

  component_list_current=component_list_head;

  for(i=1; (i<num) && (component_list_current!=NULL); i++)
    component_list_current=component_list_current->next;

  if(component_list_current==NULL) {
    fprintf(stderr,"BUG(get_component): 'num' invalid\n");
    exit(1);
  }
}


int get_named_component(char *name)
{
  component_list_current=search_component_list(name);

  return (NULL != component_list_current);
}


char* get_component_name(void)
{
  return component_list_current->name;
}


void get_component_pos(float *pos)
{
  int i;

  for(i=0; i<3; i++)
    pos[i]=component_list_current->pos[i];
}


void get_component_rot(float *rot)
{
  int i;

  for(i=0; i<9; i++)
    rot[i]=component_list_current->rot[i];
}



/************************ Draw Object List Routines ********************/

struct mc3d_draw_list *draw_list_current=NULL;

 int get_num_draw_objects(void)
{
  int i=0;
  struct mc3d_draw_list *p;

  if(component_list_current==NULL) {
    fprintf(stderr,"BUG(get_num_draw_objects): 'component_list_current' invalid\n");
    exit(1);
  }

  p=component_list_current->draw_list;
  while(p != NULL) {
    i++;
    p=p->next;
  }

  return i;
}


 void get_draw_object(int num)
{
  int i;

  if(component_list_current==NULL) {
    fprintf(stderr,"BUG(get_draw_object): 'component_list_current' invalid\n");
    exit(1);
  }

  draw_list_current=component_list_current->draw_list;

  for(i=1; (i<num) && (draw_list_current!=NULL); i++)
    draw_list_current=draw_list_current->next;

  if(draw_list_current==NULL) {
    fprintf(stderr,"BUG(get_draw_object): 'num' invalid\n");
    exit(1);
  }
}


int get_draw_object_token(void)
{
  return draw_list_current->token;
}


int get_draw_object_args(float *args)
{
  int i;

  for(i=0; i<(draw_list_current->nargs); i++)
    args[i]=draw_list_current->args[i];

  return draw_list_current->nargs;
}



/************************ Routine To Load Lists From File ********************/

/*  Parse file "file_name" for component information which is stored in lists */
int component_parser(char *file_name)
{
  FILE *file;
  char line_buffer[252];
  int nread=0, ntrace=0;


  file=fopen(file_name,"r");
  if(file == NULL) {
    printf("Input file %s not found\n",file_name);
    fclose(file);
    exit(1);
  }

  while( read_line(line_buffer, 252, file) ) {
    if(0 == strcmp("INSTRUMENT END:",line_buffer) )
      break;
    if(strlen(line_buffer) > 0) {
      parse_mc3d_line(line_buffer);
      parse_mcdisplay_line(line_buffer);
    }
  }


  while( read_line(line_buffer, 252, file) ) {
    if(0 == strcmp("LEAVE:",line_buffer) )
      ntrace++;
  }

  fclose(file);
  return ntrace;
}



/* ===================== test routines ========================= */


int get_component_num(char *comp)
{
  int i;
  char *name;
  static num_list;
  static char **name_list=NULL;

/* Load name list if this is the first call to the routine */
  if(name_list == NULL) {
    num_list=get_num_components();
    name_list=(char **)malloc( num_list*sizeof(char *) );
    for(i=0; i<num_list; i++) {
      get_component(i+1);
      name=get_component_name();
      name_list[i]=(char *)malloc( (1+strlen(name))*sizeof(char) );
      strcpy(name_list[i],name);
    }
  }

/* Return the index in the name list of the string 'comp' */
  for(i=0; i<num_list; i++) {
    if(0 == strcmp(comp,name_list[i]) )
      return i+1;
  }

/* Not in the name list, so return -1 */
  return -1;
}



int read_trace_block(struct mcstas_neutron_track *tracks, FILE *file)
{
  char line_buffer[255];
  int nread;
  char name[80];
  float x,y,z;
  int ntracks=0;

/* Skip lines until the start of a trace block */
  while( read_line(line_buffer, 252, file) ) {
    if(0 == strcmp("ENTER:",line_buffer) )
      break;
  }

/* Loop for reading header lines (give up if EOF found) */
  while( read_line(line_buffer, 252, file) ) {

/* Return with 'ntracks' if end of trace block found */
    if( 0 == strcmp("LEAVE:",line_buffer) )
      return ntracks;

/* Search for any COMP: lines */
    if( 0 == strncmp(line_buffer,"COMP: \"",7) ) {
      strcpy(name,&line_buffer[7]);
      name[strlen(name)-1]='\0';
      tracks->component=get_component_num(name);
      if(tracks->component < 0) {
        printf("BUG(read_trace_block): Unknown component name\n");
        exit(1);
      }

      if( !read_line(line_buffer, 252, file) )
        return -1;
      if( 3 != sscanf(line_buffer,"STATE: %g,%g,%g",&x,&y,&z) )
        return -1;
      tracks->x1=x;
      tracks->y1=y;
      tracks->z1=z;
      if( !read_line(line_buffer, 252, file) )
        return -1;
      if( 3 != sscanf(line_buffer,"STATE: %g,%g,%g",&x,&y,&z) )
        return -1;
      tracks->x2=x;
      tracks->y2=y;
      tracks->z2=z;
      ++ntracks;
      ++tracks;
    }

  } /* Loop back for more simulation block lines */

/* Must have found an EOF, so return -1. */
  return -1;
}


int neutron_track_parser(struct mcstas_neutron_track *tracks,
                           int first_neutron, int last_neutron,
                                                  char *file_name)
{
  FILE *file;
  int ntracks_block, ntracks, i;

  file=fopen(file_name,"r");
  if(file == NULL) {
    printf("Input file %s not found\n",file_name);
    fclose(file);
    exit(1);
  }

  for(i=1; i<first_neutron; i++)
    (void )read_trace_block(tracks, file);

  ntracks=0;
  for(i=first_neutron; i<=last_neutron; i++) {
    ntracks_block=read_trace_block(&tracks[ntracks], file);
    if(ntracks_block < 0) {
      printf("WARNING: Error occured reading neutron traces\n");
      break;
    }
    ntracks=ntracks+ntracks_block;
  }

  fclose(file);

  return ntracks;
}
