/*************************************************************************
 Contains high and low level routines for reading and parsing the
 simulation file 'mcstas.sim' and the monitor data files.

 The following are high level routines declared in 'read_mon.h' and
 designed to be called by external routines:

 int init_mon_files(int *n0, int *n1, int *n2)   parse 'mcstas.sim'
 struct mcstas_mon_file *get_2d_mon_file(int n)  parse 'n'th 2D file

 The structure 'mcstas_mon_file' is defined in 'read_mon.h'.
*************************************************************************/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mc2d.h"


/* Define structure to contain information from 'mcstas.sim' */
struct mcstas_sim_file {
    char *source;
    int trace;
    char *date;
    int ncount;
    int nmon;
    struct mcstas_mon_file mon[1000];
};

/* Instantiate the structure with 'sim_file' pointing to it */
struct mcstas_sim_file sim_file[1];



/* ####################################################################
   ##                                                                ##
   ##     Low level routines to read and parse lines from a file     ##
   ##                                                                ##
   #################################################################### */


/* Read a line from 'file' into 'line_buffer[buffer_size]', ignoring
   any non-printable characters. Returns false if EOF is encountered. */
int read_line(char *line_buffer, int buffer_size, FILE *file)
{
  char c;
  int ic, nread=0;

/* Read characters to (almost) fill line_buffer */
  while(nread < buffer_size-1) {
/* If EOF or new-line, then we are finished reading */
    ic=getc(file);
    if(ic == EOF || ic == '\n')
      break;
/* Load any printable characters into line_buffer */
    c=(char )ic;
    if(c >= ' ')
      line_buffer[nread++]=c;
  }

/* Append zero termination to line_buffer */
  line_buffer[nread]='\0';

/* Return false if a bare EOF encountered */
  if(ic == EOF && nread == 0)
    return false;

/* All's right in the world, so return true */
  return true;
}



/* Keep reading lines from 'file' until a line containing
   just 'string' is encountered                             */
int read_until_string(char *string, FILE *file)
{
  char line[80];

  while( read_line(line, 80, file) ) {
    if( 0 == strcmp(line,string) )
      return true;
  }
  return false;
}


/* If the string 'line' begins with 'identifier' then copy
   the remainder of 'line' to the new string at *p_dest     */
void parse_string_line(char **p_dest, char *identifier, char *line)
{
  int ilen=strlen(identifier);

  if( 0 == strncmp(line,identifier,ilen) ) {
    *p_dest=(char *)malloc( strlen(&line[ilen])+1 );
    strcpy(*p_dest,&line[ilen]);
  }
}



/* Same as 'parse_string_line()' except we strip off any
   quotations marks surrounding the new string               */
void parse_quote_line(char **p_dest, char *identifier, char *line)
{
  int ilen=strlen(identifier);

  if( 0 == strncmp(line,identifier,ilen) ) {
/* Skip ' or " at beginning of string */
    if(line[ilen] == '\'' || line[ilen] == '"')
      ++ilen;
/* Shorten string if it ends with a ' or " */
    if(line[strlen(line)-1] == '\'' || line[strlen(line)-1] == '"')
      line[strlen(line)-1]='\0';
/* Allocate space for resulting string and copy to *p_dest */
    *p_dest=(char *)malloc( strlen(&line[ilen])+1 );
    strcpy(*p_dest,&line[ilen]);
  }
}


/* Try to read 'nfloats' real values into 'floats[]' from the
   string 'line'.  Return the actual number read.               */
int read_line_of_floats(float *floats, int nfloats, char *line)
{
  int i,nread;

  for(i=0, nread=0; nread<nfloats; nread++) {
/* Set i to next non-space character */
    while(line[i] <= ' ')
      i++;
/* Attempt to read value starting from i */
    if(1 != sscanf(&line[i],"%f",(floats+nread)) )
      break;
/* Set i to next space character after istart */
    while(line[i] > ' ')
      i++;
  }

  return nread;
}



/* ####################################################################
   ##                                                                ##
   ##      Medium level routines to parse blocks in 'mcstas.sim'     ##
   ##                                                                ##
   #################################################################### */

int read_instrument_block(mcstas_sim_file *sim, FILE *file)
{
  char line[80];

/* Skip lines until the start of the instrument block */
  if( !read_until_string("begin instrument",file) )
    return false;

/* Loop for reading header lines (give up if EOF found) */
  while( read_line(line, 80, file) ) {

/* Return true if end of instrument block found */
    if( 0 == strcmp(line,"end instrument") )
      return true;

/* Parse lines */
    parse_string_line(&(sim->source),"  Instrument-source: ",line);

  } /* Loop back for more instrument block lines */

/* Must have found an EOF, so return false. */
  return false;
}



int read_simulation_block(mcstas_sim_file *sim, FILE *file)
{
  char line[80];
  int n;

/* Skip lines until the start of the simulation block */
  if( !read_until_string("begin simulation",file) )
    return false;

/* Loop for reading header lines (give up if EOF found) */
  while( read_line(line, 80, file) ) {

/* Return true if end of simulation block found */
    if( 0 == strcmp(line,"end simulation") )
      return true;

/* Parse lines */
    parse_string_line(&(sim->date),"  Date: ",line);
    if( 1 == sscanf(line,"  Ncount: %d",&n) ) {
      sim->ncount=n;
    }

  } /* Loop back for more simulation block lines */

/* Must have found an EOF, so return false. */
  return false;
}



int read_mon_block_2d(mcstas_mon_file *mon, FILE *file)
{
  char line[80], string[80];
  int n;
  float f1,f2,f3,f4;

  mon->itype=2;

/* Loop for reading header lines (give up if EOF found) */
  while( read_line(line, 80, file) ) {

/* Return true if end of mon block found */
    if( 0 == strcmp(line,"end data") )
      return true;

/* Parse lines for various strings */
    parse_string_line(&(mon->component),"  component: ",line);

/* Parse lines for various strings surrounded by quotes */
    parse_quote_line(&(mon->file),"  filename: ",line);
    parse_quote_line(&(mon->x_label),"  xlabel: ",line);
    parse_quote_line(&(mon->y_label),"  ylabel: ",line);

/* Parse xylimits lines */
    if( 4 == sscanf(line,"  xylimits: %f %f %f %f",&f1,&f2,&f3,&f4) ) {
      mon->xlo=f1;
      mon->xhi=f2;
      mon->ylo=f3;
      mon->yhi=f4;
    }

  } /* Loop back for more mon block lines */

/* Must have found an EOF, so return false. */
  return false;
}





int read_mon_block_1d(mcstas_mon_file *mon, FILE *file)
{
  char line[80], string[80];
  int n;
  float f1,f2,f3,f4;

  mon->itype=1;

/* Loop for reading header lines (give up if EOF found) */
  while( read_line(line, 80, file) ) {

/* Return true if end of mon block found */
    if( 0 == strcmp(line,"end data") )
      return true;

/* Parse lines for various strings */
    parse_string_line(&(mon->component),"  component: ",line);

/* Parse lines for various strings surrounded by single-quotes */
    parse_quote_line(&(mon->file),"  filename: ",line);
    parse_quote_line(&(mon->x_label),"  xlabel: ",line);
    parse_quote_line(&(mon->y_label),"  ylabel: ",line);

/* Parse xlimits lines */
    if( 2 == sscanf(line,"  xlimits: %f %f",&f1,&f2) ) {
      mon->xlo=f1;
      mon->xhi=f2;
    }

  } /* Loop back for more mon block lines */

/* Must have found an EOF, so return false. */
  return false;
}



int read_mon_block_0d(mcstas_mon_file *mon, FILE *file)
{
  char line[80];
  char string[80];

  mon->itype=0;

/* Loop for reading header lines (give up if EOF found) */
  while( read_line(line, 80, file) ) {

/* Return true if end of mon block found */
    if( 0 == strcmp(line,"end data") )
      return true;

/* Parse lines for various strings */
    parse_string_line(&(mon->component),"  component: ",line);

  } /* Loop back for more mon block lines */

/* Must have found an EOF, so return false. */
  return false;
}



int read_mon_block(mcstas_mon_file *mon, FILE *file)
{
  char line[80];
  int n, itype;

/* Skip lines until the start of the simulation block */
  if( !read_until_string("begin data",file) )
    return false;

/* Next line must be the "type" line */
  if( read_line(line, 80, file) ) {
    if( 1 == sscanf(line,"  type: array_%dd",&itype) ) {
/* Determine nx & ny if necessary, then jump to
   routine to process the headers for 0, 1 or 2 D data */
      if(itype == 0)
        return read_mon_block_0d(mon, file);
      else if(itype == 1) {
        if( 1 != sscanf(line,"  type: array_1d(%d)",&mon->nx) )
          return false;
        return read_mon_block_1d(mon, file);
      }
      else if(itype == 2) {
        if( 2 != sscanf(line,"  type: array_2d(%d,%d)",&mon->nx,&mon->ny) )
          return false;
        return read_mon_block_2d(mon, file);
      }
    }
  }

/* Something wrong with the type line, so return false. */
  return false;
}




/* ####################################################################
   ##                                                                ##
   ##        Medium level routines to parse monitor data files       ##
   ##                                                                ##
   #################################################################### */


int read_data_file(struct mcstas_mon_file *mon, FILE *file)
{
  int istatus, n1,n2, ix,iy, nread;
  char line[64000], string[80];
  float *floats;

/*************** Read and decode header comments ***********/
/* Keep reading lines until an EOF is encountered */
  while( (istatus=read_line(line, 64000, file)) ) {

/* Line not a comment, so break loop */
    if(line[0] != '#')
      break;

/* Check line for dimensions of array, if present */
    if( 2 == sscanf(line,"# type: array_2d(%d,%d)",&n1,&n2) ) {
      if( (mon->nx !=n1) || (mon->ny != n2) ) {
        printf("ERROR:  Inconsistent array size in header\n");
        return false;
      }
    }

/* Check other lines for consistency */
    if( 0 == strncmp(line,"# Instrument-source: ",21) ) {
      if(0 != strcmp(mon->source,&line[21]) ) {
        printf("ERROR:  Inconsistent instrument source in header\n");
        return false;
      }
    }
/* --------- some day check if date is before or after ---- */
//    if( 0 == strncmp(line,"# Date: ",8) ) {
//      if(0 != strcmp(mon->date,&line[8]) ) {
//        printf("ERROR:  Inconsistent date in header\n");
//        return false;
//      }
//    }
    if( 0 == strncmp(line,"# component: ",13) ) {
      if(0 != strcmp(mon->component,&line[13]) ) {
        printf("ERROR:  Inconsistent component in header\n");
        return false;
      }
    }

  }   /* End of loop reading header lines */

/* Check if EOF was read, then return false */
  if( !istatus ) {
    printf("ERROR: Unexpected EOF in data file\n");
    return false;
  }

/********* Read 2D data array *********/
/* First allocate space for data array */
  mon->data=(float *)malloc((mon->nx)*(mon->ny)*sizeof(float));

/* Start reading 'ny' lines */
  for(iy=0; iy<(mon->ny); iy++) {

/* Read a new line, except for the first time! */
    if(iy != 0) {
      if( !read_line(line, 64000, file) ) {
        printf("ERROR: Unexpected EOF in data file\n");
        return false;
      }
    }

/* Decode a line of data into floats */
    nread=read_line_of_floats( (mon->data+iy*(mon->nx)),
						(mon->nx), line);

/* Check we read the right number of floats */
    if(nread != (mon->nx)) {
      printf("ERROR: Short data line\n");
      return false;
    }

  } /* end loop for reading 'ny' lines */

  return true;
}




/* ####################################################################
   ##                                                                ##
   ##          High level routines to parse monitor files            ##
   ##                                                                ##
   #################################################################### */

/* Parse the simulation file 'mcstas.sim' copying relavent data
   to 'sim_file' and returning true if successfully parsed.
   The parameters n0,n1,n2 return with the number of 0D,1D,2D
   monitors found in the simulation.                            */
int init_mon_files(int *n0, int *n1, int *n2)
{
  int i;
  FILE *file;
  
/* Open 'mcstas.sim' */
  file=fopen("mcstas.sim","r");
  if(file == NULL) {
    printf("Input file not found\n");
    return false;
  }

/* Parse the instrument and simulation blocks */
  if( !read_instrument_block(sim_file, file) ) {
    printf("ERROR: Unable to read 'instrument' block\n");
    return false;
  }
  if( !read_simulation_block(sim_file, file) ) {
    printf("ERROR: Unable to read 'simulation' block\n");
    return false;
  }

/* Print out the information */
  printf("Instrument source file = %s\n",sim_file->source);
  printf("Date = %s\n",sim_file->date);
  printf("Ncount = %d\n",sim_file->ncount);
  printf("trace = %d\n",sim_file->trace);

/* Start reading monitor blocks, copy the relevant information
   for monitor 'i' into the 'sim_file->mon[i]' structure, and
   count the number of 0D,1D,2D monitor types.                 */
  *n0=*n1=*n2=0;
  for(i=0; i<1000; i++) {
    if( !read_mon_block(&(sim_file->mon[i]), file) )
      break;
    (sim_file->mon[i]).source=sim_file->source;
    (sim_file->mon[i]).date=sim_file->date;
    if( 2 == (sim_file->mon[i]).itype )
      ++(*n2);
    else if( 1 == (sim_file->mon[i]).itype )
      ++(*n1);
    else
      ++(*n0);
    sim_file->nmon=i;
  }

/* Warn about limit of 1000 components */
  if(sim_file->nmon == 1000)
    printf("WARNING: Have you used more than 1000 components!!!\n");

/* All over, close file and return with true */
  fclose(file);
  return true;
}



/* Return the monitor file structure for the 'mon_2d_wanted'
   occurence of a 2D monitor.  Returns NULL if an error occurs.  */
struct mcstas_mon_file *get_2d_mon_file(int mon_2d_wanted)
{
  int i, mon_2d=0;
  FILE *file;

/* Loop through monitors to find the next 2D monitor */
  for(i=0; i<(sim_file->nmon); i++) {
    if( 2 == (sim_file->mon[i]).itype ) {

/* Just count 2D monitor until we get the one we want */
      if(mon_2d++ == mon_2d_wanted) {

/* Print out monitor file name and open file */
        printf("Opening monitor file=%s\n",(sim_file->mon[i]).file);
        file=fopen((sim_file->mon[i]).file,"r");
        if(file == NULL) {
          printf("ERROR: File not found\n");
          return NULL;
        }

/* Read data into monitor record, then close file */
        if( !read_data_file(&(sim_file->mon[i]),file) ) {
          fclose(file);
          return NULL;
        }
        fclose(file);

/* Return with the structure containing the monitor data */
        return &(sim_file->mon[i]);

      } /* end of if() for counting 2D monitors */
    } /* end of if() for 2D monitor types */
  } /* end of loop over monitors */

/* If we got here we failed, so return NULL */
  return NULL;
}
